#include "pe_trader.h"
// #include <fcntl.h>
// #include <signal.h>
// #include <stdio.h>
// #include <stdlib.h>
// #include <string.h>
// #include <sys/types.h>
// #include <unistd.h>
#include <signal.h>
#include <unistd.h>

// #define _GNU_SOURCE
// #define _POSIX_C_SOURCE 199309L

struct Trader {
    int trader_id;
    int pipe_read_fd;
    int pipe_write_fd;
    int order_id;
};

static volatile sig_atomic_t running = 1;

void sigusr1_handler(int signum) {
    // Signal handler for SIGUSR1
}

void signal_handler(int signum) {
    // if (signum == SIGINT) {
    //     running = 0;
    // }
}

Trader *trader_create(int trader_id) {
    Trader *trader = malloc(sizeof(Trader));
    trader->trader_id = trader_id;
    trader->pipe_read_fd = -1;
    trader->pipe_write_fd = -1;
    trader->order_id = 0; // Initialize order_id to 0 instead of 1
    return trader;
}


void trader_destroy(Trader *trader) {
    if (trader->pipe_read_fd != -1) {
        close(trader->pipe_read_fd);
    }
    if (trader->pipe_write_fd != -1) {
        close(trader->pipe_write_fd);
    }
    free(trader);
}

void connect_to_pipes(Trader *trader) {
    char pipe_name[32];
    snprintf(pipe_name, sizeof(pipe_name), FIFO_EXCHANGE, trader->trader_id);
    trader->pipe_read_fd = open(pipe_name, O_RDONLY);

    if (trader->pipe_read_fd == -1) {
        perror("Error opening pipe for reading");
        exit(EXIT_FAILURE);
    }

    snprintf(pipe_name, sizeof(pipe_name), FIFO_TRADER, trader->trader_id);
    trader->pipe_write_fd = open(pipe_name, O_WRONLY);

    if (trader->pipe_write_fd == -1) {
        perror("Error opening pipe for writing");
        exit(EXIT_FAILURE);
    }
}

void handle_market_message(Trader *trader, const char *message) {
    char action[5];
    char product[32];
    int quantity;
    int price;

    sscanf(message, "MARKET %4s %31s %d %d", action, product, &quantity, &price);

    if (strcmp(action, "SELL") == 0) {
        if (quantity >= 1000) {
            running = 0;
        } else {
            char order[128];
            snprintf(order, sizeof(order), "BUY %d %s %d %d;", trader->order_id++, product, quantity, price); // Add semicolon at the end
            write(trader->pipe_write_fd, order, strlen(order));
            kill(getppid(), SIGUSR1);
        }
    }
}


void trader_run(Trader *trader) {
    connect_to_pipes(trader);

    signal(SIGUSR1, sigusr1_handler);
    signal(SIGINT, signal_handler);

    struct sigaction sa;
    sigemptyset(&sa.sa_mask);
    sa.sa_flags = SA_RESTART; // Automatically restart interrupted system calls
    sa.sa_handler = sigusr1_handler;

    if (sigaction(SIGUSR1, &sa, NULL) == -1) {
        perror("Error setting up sigaction for SIGUSR1");
        exit(EXIT_FAILURE);
    }

    // Create a signal set containing SIGUSR1
    sigset_t sigset;
    sigemptyset(&sigset);
    sigaddset(&sigset, SIGUSR1);

    // Block SIGUSR1
    if (sigprocmask(SIG_BLOCK, &sigset, NULL) == -1) {
        perror("Error blocking SIGUSR1");
        exit(EXIT_FAILURE);
    }

    char message[128];

    while (running) {
        memset(message, 0, sizeof(message));
        int ret = read(trader->pipe_read_fd, message, sizeof(message) - 1);

        if (ret > 0) {
            if (strstr(message, "MARKET ") == message) {
                handle_market_message(trader, message);
            }
        } else {
            break;
        }

        // Wait for SIGUSR1
        int sig;
        if (sigwait(&sigset, &sig) == -1) {
            perror("Error waiting for SIGUSR1");
            exit(EXIT_FAILURE);
        }
    }
}

int main(int argc, char *argv[]) {
    if (argc != 2) {
        fprintf(stderr, "Usage: %s trader_id\n", argv[0]);
        exit(EXIT_FAILURE);
    }

    int trader_id = atoi(argv[1]);
    Trader *trader = trader_create(trader_id);
    trader_run(trader);
    trader_destroy(trader);

    return 0;
}
