#include <stdlib.h>
#include <malloc.h>
#include <stdbool.h>

//Alternate solution to q2, involving a different typed sentinal node

//This solution allows us to use some form of type safety, we usually never want to access our linked list from outside our sentinal value.

//Since we know that our head is guaranteed to be of the type "sentinal" we can say that if we attempt to give a bad head 
//node to our program then our compiler will throw an error.

//However in the event we do (ie. node_next) we will have to use a union of both types, as we are usually unsure if we want our next node or next sentinal.
//This is a sort of pseudo-polymorphism, as we can pass in a struct with an ambigous type (similar to java interfaces almost)

struct node;
struct sentinal;


typedef struct sentinal{
	struct node* next; // our next pointer is at an offset of 0/16
	int length;

} sentinal;

typedef struct node{
	int data;
	struct node* next; //our next pointer is at an offset of 8/16

} node;


//if our next pointers were at the same offset, we could probably cast a sentinal directly to a node, however I do not recommend due to memory shennanigans.


//allows us to pass in both nodes and sentinals as a sort of pseudo-interface for both.
typedef union nodesent{
	struct node n;
	struct sentinal s;

} nodesent;

//standard list initialisation
sentinal* list_init(int value){
	node* n = calloc(1,sizeof(node));
	sentinal* s  = calloc(1,sizeof(sentinal));
	s->next = n;
	s->length = 1;
	n->data = value;
	return s;
}

//we can only get our list_end from a sentinal
node* list_end(sentinal* head){
	if (!head->next){
		return NULL;
	}
	node* n = head->next;
	while(n->next != NULL){
		n = n->next;
	}
	return n;
}




//we should generally only want to add stuff from our sentinal value.
node* list_add(sentinal* head, int value){
	node* n = list_end(head);
	//if our first node is null:
	if (n == NULL){
		node* toAdd = calloc(1,sizeof(node));
		toAdd->data = value;
		head->next = toAdd;
		head->length++;
		return toAdd;
	}
	node* toAdd = calloc(1,sizeof(node));
	n->next = toAdd;
	toAdd->data = value;
	head->length++;
	return toAdd;
}



//adds a node immediately after our head node (CAN ONLY BE DONE ON HEAD NODE).
node* add_next_head(sentinal* head, int val){
    node* toAdd = (node*) calloc(1,sizeof(node));
    toAdd->data = val;
    //propagate our node value.
    toAdd->next = head->next;
    head->next = toAdd;
    head->length++;
    return toAdd;
}


//adds a node immediately after our current node (CAN ONLY BE DONE ON A DIFFERENT NODE).
node* add_next_node(node* node, int val){
    struct node* toAdd;
    toAdd = (struct node*) calloc(1,sizeof(node));
    toAdd->data = val;
    //propagate our node value.
    toAdd->next = node->next;
    node->next = toAdd;
    //NOTE: WE CANNOT INCREASE OUR LENGTH OF OUR LIST WITHOUT A HEAD POINTER.
    return toAdd;
}

//adds a node immediately after our current node (CAN BE DONE ON BOTH).
node* add_next_poly(nodesent* ns, bool is_sentinal,int val){
    if (is_sentinal){
    	return add_next_head(&(ns->s),val);
    }
    return add_next_node(&(ns->n),val);
}



//we should only delete stuff outside of our sentinal value.
int list_delete(sentinal* head, node* n){
	if (head->next == n){
		node* ptr = n->next;
		head->next = ptr;
		free(n);
		return 0;
	}
	node* cont = head->next;
	while (cont->next && cont->next != n){
		cont = cont->next;
	}
	if (cont->next == n){
		node* ptr = n->next;
		cont->next = ptr;
		head->length--;
		free(n);
		return 0;
	}
	return 1;
}

//we don't know if we are passing in a node or a sentinal, 
//we need to pass in a second value to confirm if our node is a sentinal or a regular node.
//1 if our value in n is a sentinal, 0 if otherwise.
node* list_next(union nodesent* n, int is_sentinal){
	if (is_sentinal){
		return (n->s).next;
	}
	return (n->n).next;

}
int list_free(sentinal* s){

	node* next_node = s->next;
	free(s);
	while (next_node){
		node* temp = next_node->next;
		free(next_node);
		next_node = temp;
	}
	return 0;
}


void print_all(sentinal* head){
	node* n;
	n = head->next;
	//since we have a sentinal struct value we are allowed the luxury of having a length variable.
	printf("The size of our linked list is: %d\n",head->length);
	while (n != NULL){
		printf("%d ",n->data);
		n = n->next;
	}
	printf("\n");
}


int main(){
	sentinal* s = list_init(5);
	list_add(s,100);
	node* toRem = list_add(s,50);
	list_add(s,10);
	list_add(s,20);
	
	node* ln_test = list_next((nodesent*)s,1);
	ln_test = list_next((nodesent*)ln_test, 0);

	//we have to increment the length because our node_next and node_poly don't increment our length
	add_next_node(toRem, 25);
	s->length++;

	add_next_poly((nodesent*) toRem, 0, 60);
	s->length++;
	
	add_next_head(s, 32);
	add_next_poly((nodesent*) s, 1, 64);	
	
	printf("%d\n",ln_test->data);
	 
	printf("%d\n",ln_test->next->data);
	list_delete(s, toRem);
	printf("%d\n",ln_test->next->data);
	print_all(s);
	list_free(s);
	
}
