#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <sys/wait.h>
#include <fcntl.h>
#include <limits.h>
#include "pe_exchange.h"

typedef struct Order {
    int order_id;
    int trader_id;
    char order_type;
    char product_symbol[32];
    int quantity;
    int price;
    struct Order *next;
} Order;

typedef struct {
    char symbol[32];
    Order *buy_orders;
    Order *sell_orders;
} Product;

typedef struct {
    int trader_count;
    pid_t *trader_pids;
    char **trader_binaries;
    int *fd_exchange_to_trader;
    int *fd_trader_to_exchange;
    int *trader_pipes;
    int product_count;
    Product *products;
    Product *order_book;
} Exchange;




void load_products(Exchange *exchange, const char *product_file) {
    FILE *file = fopen(product_file, "r");
    if (!file) {
        perror("Error opening product file");
        exit(EXIT_FAILURE);
    }

    // Count the number of lines (products) in the file
    int product_count = 0;
    char c;
    while ((c = fgetc(file)) != EOF) {
        if (c == '\n') {
            product_count++;
        }
    }

    // Allocate memory for products and rewind the file
    exchange->product_count = product_count;
    exchange->products = calloc(product_count, sizeof(Product));
    fseek(file, 0, SEEK_SET);

    // Read product symbols from the file
    for (int i = 0; i < product_count; i++) {
        fscanf(file, "%31s", exchange->products[i].symbol);
    }

    fclose(file);
}

Exchange *exchange_create(const char *product_file, int trader_count, char **trader_binaries) {
    Exchange *exchange = malloc(sizeof(Exchange));
    exchange->trader_count = trader_count;
    exchange->trader_pids = calloc(trader_count, sizeof(pid_t));
    exchange->fd_exchange_to_trader = calloc(trader_count, sizeof(int));
    exchange->fd_trader_to_exchange = calloc(trader_count, sizeof(int));
    load_products(exchange, product_file);
    return exchange;
}

void launch_trader_processes(Exchange *exchange, char **trader_binaries) {
    char pipe_exchange_name[32];
    char pipe_trader_name[32];

    for (int i = 0; i < exchange->trader_count; i++) {
        snprintf(pipe_exchange_name, sizeof(pipe_exchange_name), "/tmp/pe_exchange_%d", i);
        snprintf(pipe_trader_name, sizeof(pipe_trader_name), "/tmp/pe_trader_%d", i);

        mkfifo(pipe_exchange_name, 0666);
        mkfifo(pipe_trader_name, 0666);

        pid_t pid = fork();
        if (pid == 0) {
            char trader_id_str[16];
            snprintf(trader_id_str, sizeof(trader_id_str), "%d", i);
            execl(trader_binaries[i], trader_binaries[i], trader_id_str, NULL);
            perror("Error launching trader process");
            exit(EXIT_FAILURE);
        } else if (pid > 0) {
            exchange->trader_pids[i] = pid;
            exchange->fd_exchange_to_trader[i] = open(pipe_exchange_name, O_WRONLY);
            exchange->fd_trader_to_exchange[i] = open(pipe_trader_name, O_RDONLY);
        } else {
            perror("Error forking trader process");
            exit(EXIT_FAILURE);
        }
    }
}


void process_orders(Exchange *exchange) {
    // Read incoming orders from named pipes
    char buf[256];
    for (int i = 0; i < exchange->trader_count; i++) {
        // Read from the pipe for each trader
        int bytes_read = read(exchange->trader_pipes[i], buf, 256);
        buf[bytes_read] = '\0';

        // Parse the order from the buffer
        Order order;
        parse_order(buf, &order);

        // Validate the order and update the order book
        int result = validate_and_update_order_book(&exchange->order_book, &order);

        // Send appropriate messages to traders
        send_order_status_message(exchange, i, &order, result);
    }
}

void match_orders(Exchange *exchange) {
    // Match BUY and SELL orders based on price-time priority
    // Iterate through the order book and try to match orders
    for (int i = 0; i < exchange->product_count; i++) {
        Product *product = &exchange->products[i];
        match_orders_for_product(exchange, product);
    }
}

void handle_disconnections(Exchange *exchange) {
    for (int i = 0; i < exchange->trader_count; i++) {
        int status;
        pid_t result = waitpid(exchange->trader_pids[i], &status, WNOHANG);

        if (result == 0) {
            // Trader is still running
            continue;
        }

        if (result == -1) {
            perror("Error checking trader process status");
            exit(EXIT_FAILURE);
        }

        // Handle disconnection and cleanup resources
        if (WIFEXITED(status)) {
            printf("Trader %d disconnected with status %d\n", i, WEXITSTATUS(status));
            cleanup_trader_resources(exchange, i);
        } else if (WIFSIGNALED(status)) {
            printf("Trader %d terminated by signal %d\n", i, WTERMSIG(status));
            cleanup_trader_resources(exchange, i);
        }
    }
}


//debug***
void send_market_message_to_traders(Exchange *exchange, const char *market_message) {
    for (int i = 0; i < exchange->trader_count; i++) {
        if (exchange->trader_pids[i] > 0) {
            printf("[PEX-Milestone] Exchange -> Trader: %s;\n", market_message);
            write(exchange->fd_exchange_to_trader[i], market_message, strlen(market_message));
            write(exchange->fd_exchange_to_trader[i], "\0", 1); // Add this line to send the null character
        }
    }
}

void broadcast_market_message(Exchange *exchange, const char *message) {
    for (int i = 0; i < exchange->trader_count; i++) {
        printf("[PEX-Exchange] Sending message to trader %d: %s\n", i, message);
        write(exchange->fd_exchange_to_trader[i], message, strlen(message));
        write(exchange->fd_exchange_to_trader[i], "\0", 1); // Add this line to send the null character
        printf("[PEX-Exchange] Sent message to trader %d: %s\n", i, message);
    }
}

void send_market_sell_message(Exchange *exchange) {
    // Replace the symbol and price with the values you want to send
    char market_sell_message[] = "MARKET SELL GPU 50 3499";
    for (int i = 0; i < exchange->trader_count; i++) {
        if (exchange->trader_pids[i] > 0) {
            printf("[PEX-Milestone] Exchange -> Trader: %s;\n", market_sell_message);
            write(exchange->fd_exchange_to_trader[i], market_sell_message, strlen(market_sell_message));
            write(exchange->fd_exchange_to_trader[i], "\0", 1); // Add this line to send the null character
        }
    }
}

//***debug



void exchange_run(Exchange *exchange, char **trader_binaries) {
    launch_trader_processes(exchange, trader_binaries);

    // Broadcast the "MARKET OPEN" message to traders
    broadcast_market_message(exchange, "MARKET OPEN");

    // Send the "MARKET SELL" message to traders
    send_market_sell_message(exchange);

    int connected_traders = exchange->trader_count;

    while (connected_traders > 0) {
        process_orders(exchange);
        match_orders(exchange);

        int initial_connected_traders = connected_traders;
        handle_disconnections(exchange);
        
        // Update the connected traders count after handling disconnections
        connected_traders = 0;
        for (int i = 0; i < exchange->trader_count; i++) {
            if (exchange->trader_pids[i] != -1) {
                connected_traders++;
            }
        }

        if (connected_traders != initial_connected_traders) {
            printf("Connected traders: %d\n", connected_traders);
        }
    }

    // Print final summary and clean up resources
    print_summary(exchange);
}



void exchange_destroy(Exchange *exchange) {
    // Free resources and destroy the exchange

    // Close and remove named pipes
    for (int i = 0; i < exchange->trader_count; i++) {
        char pipe_name_exchange[PATH_MAX];
        char pipe_name_trader[PATH_MAX];

        snprintf(pipe_name_exchange, sizeof(pipe_name_exchange), "/tmp/pe_exchange_%d", i);
        snprintf(pipe_name_trader, sizeof(pipe_name_trader), "/tmp/pe_trader_%d", i);

        close(exchange->fd_exchange_to_trader[i]);
        close(exchange->fd_trader_to_exchange[i]);

        unlink(pipe_name_exchange);
        unlink(pipe_name_trader);
    }

    // Free allocated arrays
    free(exchange->trader_pids);
    free(exchange->fd_exchange_to_trader);
    free(exchange->fd_trader_to_exchange);
    free(exchange->products);
    free(exchange->order_book);

    // Destroy the Exchange structure
    free(exchange);
}


int main(int argc, char *argv[]) {
    if (argc < 3) {
        fprintf(stderr, "Usage: %s <product_file> <trader_0> <trader_1> ... <trader_n>\n", argv[0]);
        return 1;
    }

    Exchange *exchange = exchange_create(argv[1], argc - 2, &argv[2]);
    exchange_run(exchange, &argv[2]);
    exchange_destroy(exchange);

    return 0;
}
